<?php
namespace CoursectorElementor\Widgets;

use Elementor\Widget_Base;
use Elementor\Controls_Manager;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * Elementor Portfolio Classic
 *
 * Elementor widget for portfolio posts
 *
 * @since 1.0.0
 */
class Coursector_Course_Grid extends Widget_Base {

	/**
	 * Retrieve the widget name.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'coursector-course-grid';
	}

	/**
	 * Retrieve the widget title.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return __( 'Course Grid', 'coursector-elementor' );
	}

	/**
	 * Retrieve the widget icon.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'eicon-posts-grid';
	}

	/**
	 * Retrieve the list of categories the widget belongs to.
	 *
	 * Used to determine where to display the widget in the editor.
	 *
	 * Note that currently Elementor supports only one category.
	 * When multiple categories passed, Elementor uses the first one.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'coursector-theme-widgets-category' ];
	}

	/**
	 * Retrieve the list of scripts the widget depended on.
	 *
	 * Used to set scripts dependencies required to run the widget.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return array Widget scripts dependencies.
	 */
	public function get_script_depends() {
		return [ 'imagesloaded', 'coursector-elementor' ];
	}
	
	/**
	 * Retrieve course categories
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return array Course categories
	 */
	public function get_course_categories() {
		//Get all categories
		$categories_arr = get_terms('course_category', 'hide_empty=0&hierarchical=0&parent=0&orderby=menu_order');
		$tg_categories_select = array();
		
		foreach ($categories_arr as $cat) {
			$tg_categories_select[$cat->term_id] = $cat->name;
		}

		return $tg_categories_select;
	}
	
	/**
	 * Retrieve course tags
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return array Course tags
	 */
	public function get_course_tags() {
		//Get all tags
		$tags_arr = get_terms('course_tag', 'hide_empty=0&hierarchical=0&parent=0&orderby=menu_order');
		$tg_tags_select = array();
		
		foreach ($tags_arr as $tag) {
			$tg_tags_select[$tag->term_id] = $tag->name;
		}

		return $tg_tags_select;
	}

	/**
	 * Register the widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 1.0.0
	 *
	 * @access protected
	 */
	protected function _register_controls() {
		$this->start_controls_section(
			'section_content',
			[
				'label' => __( 'Content', 'coursector-elementor' ),
			]
		);
		
		$this->add_control(
			'grid_template',
			[
				'label' => __( 'Grid Template', 'coursector-elementor' ),
				'type' => Controls_Manager::SELECT,
				'default' => 1,
			    'options' => [
				    1 => __( 'Style 1', 'coursector-elementor' ),
				    2 => __( 'Style 2', 'coursector-elementor' ),
				    3 => __( 'Style 3', 'coursector-elementor' ),
			    ],
			]
		);
		
		$this->add_control(
			'course_category',
			[
				'label' => __( 'Filter by Categories', 'coursector-elementor' ),
				'type' => Controls_Manager::SELECT2,
			    'options' => $this->get_course_categories(),
			    'multiple' => true,
			]
		);
		
		$this->add_control(
			'course_tag',
			[
				'label' => __( 'Filter by Tags', 'coursector-elementor' ),
				'type' => Controls_Manager::SELECT2,
			    'options' => $this->get_course_tags(),
			    'multiple' => true,
			]
		);
		
		$this->add_control(
			'sort_by',
			[
				'label' => __( 'Sort By', 'coursector-elementor' ),
				'type' => Controls_Manager::SELECT,
				'default' => 'default',
			    'options' => [
				    'default' => __( 'Default', 'coursector-elementor' ),
				    'random' => __( 'Random', 'coursector-elementor' ),
				    'published' => __( 'Published Date', 'coursector-elementor' ),
			     	'title' => __( 'Title', 'coursector-elementor' ),
			     	'price_low' => __( 'Price (Low to High)', 'coursector-elementor' ),
			     	'price_high' => __( 'Price (High to Low)', 'coursector-elementor' ),
			    ],
			]
		);

		$this->add_control(
		    'posts_per_page',
		    [
		        'label' => __( 'Posts Per Page', 'coursector-elementor' ),
		        'type' => Controls_Manager::SLIDER,
		        'default' => [
		            'size' => 6,
		        ],
		        'range' => [
		            'px' => [
		                'min' => -1,
		                'max' => 100,
		                'step' => 1,
		            ]
		        ],
		    ]
		);
		
		$this->add_control(
		    'columns',
		    [
		        'label' => __( 'Columns', 'coursector-elementor' ),
		        'type' => Controls_Manager::SLIDER,
		        'default' => [
		            'size' => 3,
		        ],
		        'range' => [
		            'px' => [
		                'min' => 2,
		                'max' => 4,
		                'step' => 1,
		            ]
		        ],
		    ]
		);
		
		$this->add_control(
			'spacing',
			[
				'label' => __( 'Column Spacing', 'coursector-elementor' ),
				'type' => Controls_Manager::SWITCHER,
				'default' => 'yes',
				'label_on' => __( 'Yes', 'coursector-elementor' ),
				'label_off' => __( 'No', 'coursector-elementor' ),
				'return_value' => 'yes',
			]
		);
		
		$this->add_control(
			'show_pagination',
			[
				'label' => __( 'Show Pagination', 'coursector-elementor' ),
				'type' => Controls_Manager::SWITCHER,
				'default' => 'yes',
				'label_on' => __( 'Yes', 'coursector-elementor' ),
				'label_off' => __( 'No', 'coursector-elementor' ),
				'return_value' => 'yes',
			]
		);
		
		$this->add_control(
			'hover_effect',
			[
				'label' => __( 'Hover Effect', 'coursector-elementor' ),
				'type' => Controls_Manager::SWITCHER,
				'default' => 'yes',
				'label_on' => __( 'Yes', 'coursector-elementor' ),
				'label_off' => __( 'No', 'coursector-elementor' ),
				'return_value' => 'yes',
			]
		);
		
		$this->end_controls_section();
		
		$this->start_controls_section(
			'section_meta',
			[
				'label' => __( 'Meta Data', 'coursector-elementor' ),
			]
		);
		
		$this->add_control(
		    'excerpt_length',
		    [
		        'label' => __( 'Excerpt Length', 'coursector-elementor' ),
		        'type' => Controls_Manager::SLIDER,
		        'default' => [
		            'size' => 90,
		        ],
		        'range' => [
		            'px' => [
		                'min' => 0,
		                'max' => 300,
		                'step' => 1,
		            ]
		        ],
		    ]
		);
		
		$this->add_control(
			'show_rating',
			[
				'label' => __( 'Show Rating', 'coursector-elementor' ),
				'type' => Controls_Manager::SWITCHER,
				'default' => 'no',
				'label_on' => __( 'Yes', 'coursector-elementor' ),
				'label_off' => __( 'No', 'coursector-elementor' ),
				'return_value' => 'yes',
			]
		);
		
		$this->add_control(
			'show_lesson',
			[
				'label' => __( 'Show Lesson', 'coursector-elementor' ),
				'type' => Controls_Manager::SWITCHER,
				'default' => 'yes',
				'label_on' => __( 'Yes', 'coursector-elementor' ),
				'label_off' => __( 'No', 'coursector-elementor' ),
				'return_value' => 'yes',
			]
		);
		
		$this->add_control(
			'show_student',
			[
				'label' => __( 'Show Student Number', 'coursector-elementor' ),
				'type' => Controls_Manager::SWITCHER,
				'default' => 'yes',
				'label_on' => __( 'Yes', 'coursector-elementor' ),
				'label_off' => __( 'No', 'coursector-elementor' ),
				'return_value' => 'yes',
			]
		);
		
		$this->add_control(
			'show_price',
			[
				'label' => __( 'Show Price', 'coursector-elementor' ),
				'type' => Controls_Manager::SWITCHER,
				'default' => 'yes',
				'label_on' => __( 'Yes', 'coursector-elementor' ),
				'label_off' => __( 'No', 'coursector-elementor' ),
				'return_value' => 'yes',
			]
		);
		
		$this->add_control(
			'show_tooltip',
			[
				'label' => __( 'Show Tooltip', 'coursector-elementor' ),
				'type' => Controls_Manager::SWITCHER,
				'default' => 'yes',
				'label_on' => __( 'Yes', 'coursector-elementor' ),
				'label_off' => __( 'No', 'coursector-elementor' ),
				'return_value' => 'yes',
			]
		);
		
		$this->end_controls_section();
		
		$this->start_controls_section(
			'section_title_style',
			array(
				'label'      => esc_html__( 'Title', 'coursector-elementor' ),
				'tab'        => Controls_Manager::TAB_STYLE,
				'show_label' => false,
			)
		);
		
		$this->add_control(
		    'title_color',
		    [
		        'label' => __( 'Title Color', 'coursector-elementor' ),
		        'type' => Controls_Manager::COLOR,
		        'default' => '#222222',
		        'selectors' => [
		            '{{WRAPPER}} .course_grid_container .portfolio_classic_grid_wrapper .card__title a' => 'color: {{VALUE}}',
		        ],
		    ]
		);
		
		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name' => 'title_typography',
				'label' => __( 'Title Typography', 'coursector-elementor' ),
				'selector' => '{{WRAPPER}} .course_grid_container .portfolio_classic_grid_wrapper .card__title',
			]
		);
		
		$this->add_control(
		    'content_bg_color',
		    [
		        'label' => __( 'Content Background Color', 'coursector-elementor' ),
		        'type' => Controls_Manager::COLOR,
		        'default' => '#ffffff',
		        'selectors' => [
		            '{{WRAPPER}} .course_grid_container .portfolio_classic_grid_wrapper .card__info' => 'background-color: {{VALUE}}',
		            '{{WRAPPER}} .course_grid_container .grid_template3 .card__info .card__title_wrapper' => 'background-color: {{VALUE}}',
		        ],
		    ]
		);
		
		$this->add_control(
		    'content_border_color',
		    [
		        'label' => __( 'Content Border Color', 'coursector-elementor' ),
		        'type' => Controls_Manager::COLOR,
		        'default' => '#ffffff',
		        'selectors' => [
		            '{{WRAPPER}} .course_grid_container .portfolio_classic_grid_wrapper' => 'border-color: {{VALUE}}',
		            '{{WRAPPER}} .course_grid_container .portfolio_classic_grid_wrapper .card__meta_wrapper' => 'border-color: {{VALUE}}',
		        ],
		    ]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Box_Shadow::get_type(),
			[
				'name' => 'card_box_shadow',
				'label' => __( 'Card Shadow', 'coursector-elementor' ),
				'selector' => '{{WRAPPER}} .course_grid_container .portfolio_classic_grid_wrapper',
			]
		);
		
		$this->end_controls_section();
		
		$this->start_controls_section(
			'section_excerpt_style',
			array(
				'label'      => esc_html__( 'Excerpt', 'coursector-elementor' ),
				'tab'        => Controls_Manager::TAB_STYLE,
				'show_label' => false,
			)
		);
		
		$this->add_control(
		    'excerpt_color',
		    [
		        'label' => __( 'Excerpt Color', 'coursector-elementor' ),
		        'type' => Controls_Manager::COLOR,
		        'default' => '#222222',
		        'selectors' => [
		            '{{WRAPPER}} .course_grid_container .portfolio_classic_grid_wrapper .card__info .card__excerpt' => 'color: {{VALUE}}',
		        ],
		    ]
		);
		
		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name' => 'excerpt_typography',
				'label' => __( 'Excerpt Typography', 'coursector-elementor' ),
				'selector' => '{{WRAPPER}} .course_grid_container .portfolio_classic_grid_wrapper .card__info .card__excerpt p',
			]
		);
		
		$this->end_controls_section();
		
		$this->start_controls_section(
			'section_meta_style',
			array(
				'label'      => esc_html__( 'Meta Data', 'coursector-elementor' ),
				'tab'        => Controls_Manager::TAB_STYLE,
				'show_label' => false,
			)
		);
		
		$this->add_control(
		    'meta_color',
		    [
		        'label' => __( 'Meta Data Color', 'coursector-elementor' ),
		        'type' => Controls_Manager::COLOR,
		        'default' => '#222222',
		        'selectors' => [
		            '{{WRAPPER}} .course_grid_container .portfolio_classic_grid_wrapper .card__info .card__meta' => 'color: {{VALUE}}',
		        ],
		    ]
		);
		
		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name' => 'meta_typography',
				'label' => __( 'Meta Data Typography', 'coursector-elementor' ),
				'selector' => '{{WRAPPER}} .course_grid_container .portfolio_classic_grid_wrapper .card__info div.card__meta',
			]
		);
		
		$this->end_controls_section();
		
		$this->start_controls_section(
			'section_pricing_style',
			array(
				'label'      => esc_html__( 'Pricing', 'coursector-elementor' ),
				'tab'        => Controls_Manager::TAB_STYLE,
				'show_label' => false,
			)
		);
		
		$this->add_control(
		    'pricing_bg_color',
		    [
		        'label' => __( 'Pricing Background Color', 'coursector-elementor' ),
		        'type' => Controls_Manager::COLOR,
		        'default' => '#3d64ff',
		        'selectors' => [
		            '{{WRAPPER}} .course_grid_container .portfolio_classic_grid_wrapper .card__price' => 'background: {{VALUE}}',
		        ],
		    ]
		);
		
		$this->add_control(
		    'pricing_color',
		    [
		        'label' => __( 'Pricing Font Color', 'coursector-elementor' ),
		        'type' => Controls_Manager::COLOR,
		        'default' => '#ffffff',
		        'selectors' => [
		            '{{WRAPPER}} .course_grid_container .portfolio_classic_grid_wrapper .card__price' => 'color: {{VALUE}}',
		        ],
		    ]
		);
		
		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name' => 'pricing_typography',
				'label' => __( 'Pricing Typography', 'coursector-elementor' ),
				'selector' => '{{WRAPPER}} .course_grid_container .portfolio_classic_grid_wrapper .card__price',
			]
		);
		
		$this->end_controls_section();
	}
	

	/**
	 * Render the widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 *
	 * @access protected
	 */
	protected function render() {
		include(COURSECTOR_ELEMENTOR_PATH.'templates/course-grid/index.php');
	}

	/**
	 * Render the widget output in the editor.
	 *
	 * Written as a Backbone JavaScript template and used to generate the live preview.
	 *
	 * @since 1.0.0
	 *
	 * @access protected
	 */
	protected function _content_template() {
		return '';
	}
}
