<?php
namespace CoursectorElementor\Widgets;

use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use Elementor\Scheme_Color;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * Elementor Search
 *
 * Elementor widget for search field
 *
 * @since 1.0.0
 */
class Coursector_Search extends Widget_Base {

	/**
	 * Retrieve the widget name.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'coursector-search';
	}

	/**
	 * Retrieve the widget title.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return __( 'Search Bar', 'coursector-elementor' );
	}

	/**
	 * Retrieve the widget icon.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'eicon-search';
	}

	/**
	 * Retrieve the list of categories the widget belongs to.
	 *
	 * Used to determine where to display the widget in the editor.
	 *
	 * Note that currently Elementor supports only one category.
	 * When multiple categories passed, Elementor uses the first one.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'coursector-theme-widgets-category' ];
	}

	/**
	 * Retrieve the list of scripts the widget depended on.
	 *
	 * Used to set scripts dependencies required to run the widget.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return array Widget scripts dependencies.
	 */
	public function get_script_depends() {
		return [ 'coursector-elementor' ];
	}

	/**
	 * Register the widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 1.0.0
	 *
	 * @access protected
	 */
	protected function _register_controls() {
		$this->start_controls_section(
			'section_content',
			[
				'label' => __( 'Content', 'coursector-elementor' ),
			]
		);
		
		$this->add_responsive_control(
		    'width',
		    [
		        'label' => __( 'Width', 'coursector-elementor' ),
		        'type' => Controls_Manager::SLIDER,
		        'default' => [
		            'size' => 350,
		            'unit' => 'px',
		        ],
		        'range' => [
		            'px' => [
		                'min' => 5,
		                'max' => 10000,
		                'step' => 5,
		            ]
		        ],
		        'size_units' => [ 'px' ],
		        'selectors' => [
		            '{{WRAPPER}} .tg_search_wrapper input' => 'width: {{SIZE}}{{UNIT}}',
		        ],
		    ]
		);
		
		$this->add_responsive_control(
		    'height',
		    [
		        'label' => __( 'Height', 'coursector-elementor' ),
		        'type' => Controls_Manager::SLIDER,
		        'default' => [
		            'size' => 44,
		            'unit' => 'px',
		        ],
		        'range' => [
		            'px' => [
		                'min' => 1,
		                'max' => 200,
		                'step' => 51,
		            ]
		        ],
		        'size_units' => [ 'px' ],
		        'selectors' => [
		            '{{WRAPPER}} .tg_search_wrapper .input_group input' => 'height: {{SIZE}}{{UNIT}}',
		            '{{WRAPPER}} .tg_search_wrapper .input_group .input_group_button button' => 'height: {{SIZE}}{{UNIT}}',
		        ],
		    ]
		);
		
		$this->add_responsive_control(
		    'icon_size',
		    [
		        'label' => __( 'Icon Size', 'coursector-elementor' ),
		        'type' => Controls_Manager::SLIDER,
		        'default' => [
		            'size' => 16,
		            'unit' => 'px',
		        ],
		        'range' => [
		            'px' => [
		                'min' => 1,
		                'max' => 50,
		                'step' => 1,
		            ]
		        ],
		        'size_units' => [ 'px' ],
		        'selectors' => [
		            '{{WRAPPER}} .tg_search_wrapper .input_group .input_group_button button' => 'font-size: {{SIZE}}{{UNIT}}',
		        ],
		    ]
		);
		
		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name' => 'search_typography',
				'label' => __( 'Typography', 'coursector-elementor' ),
				'selector' => '{{WRAPPER}} .tg_search_wrapper input',
			]
		);
		
		$this->add_control(
			'placeholder',
			[
				'label' => __( 'Placeholder Text', 'coursector-elementor' ),
				'type' => \Elementor\Controls_Manager::TEXT,
				'default' => __( 'Search for anything', 'coursector-elementor' ),
			]
		);
		
		$this->add_control(
			'redirect_page',
			[
				'label' => __( 'Redirect to page', 'coursector-elementor' ),
				'type' => \Elementor\Controls_Manager::SELECT2,
				'multiple' => false,
				'options' => $this->_get_pages(),
			]
		);
		
		$this->end_controls_section();
		
		$this->start_controls_section(
			'section_autocomplete',
			[
				'label' => __( 'Auto Complete', 'coursector-elementor' ),
			]
		);
		
		$this->add_control(
			'autocomplete',
			[
				'label' => __( 'Auto Complete', 'coursector-elementor' ),
				'type' => Controls_Manager::SWITCHER,
				'default' => 'yes',
				'label_on' => __( 'Yes', 'coursector-elementor' ),
				'label_off' => __( 'No', 'coursector-elementor' ),
				'return_value' => 'yes',
			]
		);
		
		$this->add_control(
			'search_lp_course',
			[
				'label' => __( 'Show only course result', 'coursector-elementor' ),
				'type' => Controls_Manager::SWITCHER,
				'default' => 'yes',
				'label_on' => __( 'Yes', 'coursector-elementor' ),
				'label_off' => __( 'No', 'coursector-elementor' ),
				'return_value' => 'yes',
			]
		);
		
		$this->end_controls_section();
		
		$this->start_controls_section(
			'section_style',
			array(
				'label'      => esc_html__( 'Styles', 'coursector-elementor' ),
				'tab'        => Controls_Manager::TAB_STYLE,
				'show_label' => false,
			)
		);
		
		$this->add_control(
		    'search_font_color',
		    [
		        'label' => __( 'Font Color', 'coursector-elementor' ),
		        'type' => Controls_Manager::COLOR,
		        'default' => '#000000',
		        'selectors' => [
		            '{{WRAPPER}} .tg_search_wrapper input' => 'color: {{VALUE}}',
		        ],
		    ]
		);
		
		$this->add_control(
		    'search_placeholder_font_color',
		    [
		        'label' => __( 'Placeholder Font Color', 'coursector-elementor' ),
		        'type' => Controls_Manager::COLOR,
		        'default' => '#999999',
		        'selectors' => [
		            '{{WRAPPER}} .tg_search_wrapper input::placeholder' => 'color: {{VALUE}}',
		        ],
		    ]
		);
		
		$this->add_control(
		    'search_bg_color',
		    [
		        'label' => __( 'Search Input Background Color', 'coursector-elementor' ),
		        'type' => Controls_Manager::COLOR,
		        'default' => '#F0F0F0',
		        'selectors' => [
		            '{{WRAPPER}} .tg_search_wrapper' => 'background-color: {{VALUE}}',
		        ],
		    ]
		);
		
		$this->add_control(
		    'search_border_color',
		    [
		        'label' => __( 'Search Input Border Color', 'coursector-elementor' ),
		        'type' => Controls_Manager::COLOR,
		        'default' => '#F0F0F0',
		        'selectors' => [
		            '{{WRAPPER}} .tg_search_wrapper' => 'border-color: {{VALUE}}',
		        ],
		    ]
		);
		
		$this->add_control(
		    'icon_color',
		    [
		        'label' => __( 'Icon Color', 'coursector-elementor' ),
		        'type' => Controls_Manager::COLOR,
		        'default' => '#ff5722',
		        'selectors' => [
		            '{{WRAPPER}} .tg_search_wrapper .input_group .input_group_button button' => 'color: {{VALUE}}',
		        ],
		    ]
		);
		
		$this->add_group_control(
			\Elementor\Group_Control_Box_Shadow::get_type(),
			[
				'name' => 'search_box_shadow',
				'label' => __( 'Search Input Shadow', 'coursector-elementor' ),
				'selector' => '{{WRAPPER}} .tg_search_wrapper',
			]
		);
		
		$this->add_control(
			'search_border_radius',
			[
				'label' => __( 'Search Input Border Radius', 'coursector-elementor' ),
				'type' => Controls_Manager::SLIDER,
				'size_units' => [ 'px' ],
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 25,
						'step' => 1,
					],
				],
				'default' => [
					'unit' => 'px',
					'size' => 5,
				],
				'selectors' => [
					'{{WRAPPER}} .tg_search_wrapper' => 'border-radius: {{SIZE}}{{UNIT}};',
				],
			]
		);
		
		$this->end_controls_section();
		
		$this->start_controls_section(
			'section_hover_style',
			array(
				'label'      => esc_html__( 'Hover Styles', 'coursector-elementor' ),
				'tab'        => Controls_Manager::TAB_STYLE,
				'show_label' => false,
			)
		);
		
		$this->add_control(
		    'search_hover_font_color',
		    [
		        'label' => __( 'Font Hover Color', 'coursector-elementor' ),
		        'type' => Controls_Manager::COLOR,
		        'default' => '#000000',
		        'selectors' => [
		            '{{WRAPPER}} .tg_search_wrapper:hover input' => 'color: {{VALUE}}',
		        ],
		    ]
		);
		
		$this->add_control(
		    'search_hover_placeholder_font_color',
		    [
		        'label' => __( 'Placeholder Hover Font Color', 'coursector-elementor' ),
		        'type' => Controls_Manager::COLOR,
		        'default' => '#999999',
		        'selectors' => [
		            '{{WRAPPER}} .tg_search_wrapper:hover input::placeholder' => 'color: {{VALUE}}',
		        ],
		    ]
		);
		
		$this->add_control(
		    'search_hover_bg_color',
		    [
		        'label' => __( 'Search Input Hover Background Color', 'coursector-elementor' ),
		        'type' => Controls_Manager::COLOR,
		        'default' => '#ffffff',
		        'selectors' => [
		            '{{WRAPPER}} .tg_search_wrapper:hover' => 'background-color: {{VALUE}}',
		        ],
		    ]
		);
		
		$this->add_control(
		    'search_hover_border_color',
		    [
		        'label' => __( 'Search Input Hover Border Color', 'coursector-elementor' ),
		        'type' => Controls_Manager::COLOR,
		        'default' => '#F0F0F0',
		        'selectors' => [
		            '{{WRAPPER}} .tg_search_wrapper:hover' => 'border-color: {{VALUE}}',
		        ],
		    ]
		);
		
		$this->add_control(
		    'icon_hover_color',
		    [
		        'label' => __( 'Icon Hover Color', 'coursector-elementor' ),
		        'type' => Controls_Manager::COLOR,
		        'default' => '#ff5722',
		        'selectors' => [
		            '{{WRAPPER}} .tg_search_wrapper:hover .input_group .input_group_button button' => 'color: {{VALUE}}',
		        ],
		    ]
		);
		
		$this->add_group_control(
			\Elementor\Group_Control_Box_Shadow::get_type(),
			[
				'name' => 'search_hover_box_shadow',
				'label' => __( 'Search Input Hover Shadow', 'coursector-elementor' ),
				'selector' => '{{WRAPPER}} .tg_search_wrapper:hover',
			]
		);
		
		$this->end_controls_section();
		
		$this->start_controls_section(
			'section_autocomplete_style',
			array(
				'label'      => esc_html__( 'Auto Complete Styles', 'coursector-elementor' ),
				'tab'        => Controls_Manager::TAB_STYLE,
				'show_label' => false,
			)
		);
		
		$this->add_control(
		    'autocomplete_font_color',
		    [
		        'label' => __( 'Auto Complete Font Color', 'coursector-elementor' ),
		        'type' => Controls_Manager::COLOR,
		        'default' => '#000000',
		        'selectors' => [
		            '{{WRAPPER}} .tg_search_wrapper .autocomplete li a' => 'color: {{VALUE}}',
		        ],
		    ]
		);
		
		$this->add_control(
		    'autocomplete_bg_color',
		    [
		        'label' => __( 'Auto Complete Background Color', 'coursector-elementor' ),
		        'type' => Controls_Manager::COLOR,
		        'default' => '#ffffff',
		        'selectors' => [
		            '{{WRAPPER}} .autocomplete ul' => 'background-color: {{VALUE}}',
		        ],
		    ]
		);
		
		$this->add_control(
		    'autocomplete_hover_font_color',
		    [
		        'label' => __( 'Auto Complete Hover Font Color', 'coursector-elementor' ),
		        'type' => Controls_Manager::COLOR,
		        'default' => '#000000',
		        'selectors' => [
		            '{{WRAPPER}} .autocomplete li:hover a' => 'color: {{VALUE}}',
		        ],
		    ]
		);
		
		$this->add_control(
		    'autocomplete_hover_bg_color',
		    [
		        'label' => __( 'Auto Complete Hover Background Color', 'coursector-elementor' ),
		        'type' => Controls_Manager::COLOR,
		        'default' => '#f9f9f9',
		        'selectors' => [
		            '{{WRAPPER}} .autocomplete li:hover a' => 'background-color: {{VALUE}}',
		        ],
		    ]
		);
		
		$this->add_group_control(
			\Elementor\Group_Control_Box_Shadow::get_type(),
			[
				'name' => 'autocomplete_box_shadow',
				'label' => __( 'Auto Complete Shadow', 'coursector-elementor' ),
				'selector' => '{{WRAPPER}} .autocomplete ul ',
			]
		);
		
		$this->end_controls_section();
	}

	/**
	 * Render the widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 *
	 * @access protected
	 */
	protected function render() {
		include(COURSECTOR_ELEMENTOR_PATH.'templates/search/index.php');
	}

	/**
	 * Render the widget output in the editor.
	 *
	 * Written as a Backbone JavaScript template and used to generate the live preview.
	 *
	 * @since 1.0.0
	 *
	 * @access protected
	 */
	protected function _content_template() {
		return '';
	}
	
	protected function _get_pages() {
		/*
			Get all pages available
		*/
		$pages = get_pages();
		$pages_select = array(
			 '' => __( 'Default Search Page', 'coursector-elementor' )
		);
		foreach($pages as $each_page)
		{
			$pages_select[$each_page->ID] = $each_page->post_title;
		}
		
		return $pages_select;
	}
}
